/*
 * Copyright (c) 1999-2001 Lutris Technologies, Inc. All Rights
 * Reserved.
 * 
 * This source code file is distributed by Lutris Technologies, Inc. for
 * use only by licensed users of product(s) that include this source
 * file. Use of this source file or the software that uses it is covered
 * by the terms and conditions of the Lutris Enhydra Development License
 * Agreement included with this product.
 * 
 * This Software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF
 * ANY KIND, either express or implied. See the License for the specific terms
 * governing rights and limitations under the License.
 * 
 * Contributor(s):
 * 
 * $Id: AbstractXmlWriter.java,v 1.1.1.1 2001/01/05 05:11:18 sese0235 Exp $
 */

package de.kxml.io;

import java.io.*;
import java.util.*;
import de.kxml.*;

/**
 * An abstract XmlWriter including namespace handling.
 */
public abstract class AbstractXmlWriter extends Writer {
    protected State current = new State(null, new PrefixMap(),    // null, null,
    null);

    /**
     * Class declaration
     *
     *
     * @author
     * @version %I%, %G%
     */
    protected class State {
	public State     prev;
	public PrefixMap prefixMap;

	// String namespace;
	// String name;
	public String    tag;    // for auto-endtag writing

	/**
	 * Constructor declaration
	 *
	 *
	 * @param prev
	 * @param prefixMap
	 * @param tag
	 *
	 * @see
	 */
	public State(State prev, PrefixMap prefixMap, 

	// String namespace, String name,
	String tag) {
	    this.prev = prev;
	    this.prefixMap = prefixMap;

	    // this.namespace = namespace;
	    // this.name = name;
	    this.tag = tag;
	}

	;
    }

    /**
     * writes an attribute with the given namespace. Only allowed
     * immediately after writeStartTag or another writeAttribute
     * call.
     */
    public void writeAttribute(String namespace, String name, 
			       String value) throws IOException {
	if (namespace == null || "".equals(namespace)) {
	    writeAttribute(name, value);
	} else {
	    String prefix = current.prefixMap.getPrefix(namespace);

	    if (prefix == null || prefix.equals("")) {
		int cnt = 0;

		do {
		    prefix = "p" + (cnt++);
		} while (current.prefixMap.getNamespace(prefix) != null);

		current.prefixMap = new PrefixMap(current.prefixMap, prefix, 
						  namespace);

		writeAttribute("xmlns:" + prefix, namespace);
	    } 

	    writeAttribute(prefix + ":" + name, value);
	} 
    } 

    /**
     * writes an attribute. Only allowed immediately after
     * writeStartTag or writeAttribute.
     */
    public abstract void writeAttribute(String name, 
					String value) throws IOException;

    /**
     * writes a start tag with the given name, using the given prefix
     * map.  This method cares about the namespace prefixes and calls
     * writeStartTag (PrefixMap prefixMap, String tag) for concrete
     * writing.
     */
    public void writeStartTag(PrefixMap prefixMap, String namespace, 
			      String name) throws IOException {

	// check if namespace is default.
	if (prefixMap == null) {
	    prefixMap = current.prefixMap;
	} 

	if (namespace == null) {
	    namespace = "";
	} 

	String prefix = prefixMap.getPrefix(namespace);

	if (prefix == null) {
	    prefixMap = new PrefixMap(prefixMap, "", namespace);
	    prefix = "";
	} 

	String    tag = prefix.length() == 0 ? name : prefix + ':' + name;
	PrefixMap oldMap = current.prefixMap;

	writeStartTag(prefixMap, tag);

	// if namespace has changed, write out changes...
	if (prefixMap != oldMap) {
	    for (Enumeration e = prefixMap.prefixEnumeration(); 
		    e.hasMoreElements(); ) {
		String p2 = (String) e.nextElement();
		String ns = prefixMap.getNamespace(p2);

		if (ns != oldMap.getNamespace(p2)) {
		    if (p2.equals("")) {
			writeAttribute("xmlns", ns);
		    } else {
			writeAttribute("xmlns:" + p2, ns);
		    }
		} 
	    } 
	} 
    } 

    /**
     * writes a start tag with the given namespace and name
     */
    public void writeStartTag(String namespace, 
			      String name) throws IOException {
	writeStartTag(null, namespace, name);
    } 

    /**
     * convenience method for writeStartTag ("", name)
     */
    public void writeStartTag(String name) throws IOException {
	writeStartTag(null, "", name);
    } 

    /**
     * abstract method that must be overwritten by
     * a method actually writing the resolved start tag
     * without namespace checking. This implementation
     * just puts the state on the stack.<br /><br />
     * 
     * Attention: The actual implementation include the
     * following line in order to
     * put the current State on the stack!<br /><br />
     * current = new State (current, prefixMap, tag);
     */
    protected abstract void writeStartTag(PrefixMap prefixMap, 
					  String tag) throws IOException;

    /**
     * @deprecated
     * Old method for writing a (possibly degenerated)
     * tag including the given attributes. Please use subsequent
     * calls of the other methods instead!
     */
    public void writeStartTag(String name, String[] attrs, 
			      boolean degenerated) throws IOException {
	writeStartTag(name);

	for (int i = 0; i < attrs.length; i += 2) {
	    writeAttribute(attrs[i], attrs[i + 1]);
	}

	if (degenerated) {
	    writeEndTag();
	} 
    } 

    /**
     * Abstract method for writing an end tag.  <b>Attention:</b>
     * Concrete implementations must pop the previous state from the
     * stack:<br /><br />
     * current = current.prev;
     */
    public abstract void writeEndTag() throws IOException;

    /**
     * writes Xml.DOCTYPE, Xml.PROCESSING_INSTRUCTION or Xml.COMMENT
     */
    public abstract void writeLegacy(int type, 
				     String text) throws IOException;
}

